#define MAXn 136 /* clathrat type II cell */

typedef struct {
  int nbr; /* neighbor index, in the rectangular basic cell */
  int x,y,z; /* position of the repeating cell with given neighbor */
} nbrs_t[4];

struct cell_s {
  int n;
  nbrs_t *nbrs;   /* indices of neighbors; if NULL then derived from distances */
  double unit;    /* scaling of a,shift,r[]; -1 = no scaling */
  double OO;      /* O-O distance (to calculate H,M,L) */
  double OOHB;    /* O-O distance limit to define H-bond */
  vector a;       /* rectangular basic cell, if unit>1 then rescaled by unit
                     NB: for monoclinic: a,b,c*sin(beta) */
  double shift;   /* shift a/c (used for the monoclinic cell) */
  vector r[MAXn]; /* [n] molecular positions, if unit>0 then in a,b,c */
};

/******************* 2D square *******************/

nbrs_t nbrs2D[1]={
  { {0,-1,0,0}, {0,1,0,0}, {0,0,-1,0}, {0,0,1,0} }
};

struct cell_s Ice2D = {
  1,
  nbrs2D,
  2.767,2.767,2.767*1.1,
  {1,1,2},
  0,
  {
    {0, 0, 0,},
  }
};

/******************* Ih *******************/
nbrs_t nbrsIh[8]={
  { {2,-1,0,0}, {1,0,-1,0}, {4,0,0,-1}, {2,0,0,0}, },
  { {3,-1,0,0}, {3,0,0,0}, {5,0,0,0}, {0,0,1,0}, },
  { {6,0,0,0}, {0,0,0,0}, {0,1,0,0}, {3,0,0,0}, },
  { {2,0,0,0}, {7,0,0,-1}, {1,0,0,0}, {1,1,0,0}, },
  { {6,-1,0,0}, {0,0,0,1}, {5,0,-1,0}, {6,0,0,0}, },
  { {1,0,0,0}, {7,0,0,0}, {7,-1,0,0}, {4,0,1,0}, },
  { {4,0,0,0}, {7,0,0,0}, {2,0,0,0}, {4,1,0,0}, },
  { {3,0,0,1}, {5,1,0,0}, {6,0,0,0}, {5,0,0,0}, },
};

struct cell_s IceIh = {
  8,
  nbrsIh,
  2.767,2.767,2.767*1.1,
  {1.63299316185545,2.82842712474619,8./3},
  0,
  {
    {0,  1./6, 3./16},
    {0,  5./6, 5./16},
    {0.5,2./6, 5./16},
    {0.5,4./6, 3./16},
    {0,  1./6,13./16},
    {0,  5./6,11./16},
    {0.5,2./6,11./16},
    {0.5,4./6,13./16}
  }
};

/******************* Ic *******************/
nbrs_t nbrsIc[8]={
  { {2,0,0,0}, {3,-1,-1,0}, {7,0,-1,-1}, {6,-1,0,-1}, },
  { {6,0,0,-1}, {2,0,0,0}, {3,0,0,0}, {7,0,0,-1}, },
  { {5,0,0,0}, {4,0,0,0}, {1,0,0,0}, {0,0,0,0}, },
  { {4,1,0,0}, {1,0,0,0}, {0,1,1,0}, {5,0,1,0}, },
  { {6,-1,0,0}, {7,0,0,0}, {2,0,0,0}, {3,-1,0,0}, },
  { {3,0,-1,0}, {7,0,-1,0}, {6,0,0,0}, {2,0,0,0}, },
  { {1,0,0,1}, {5,0,0,0}, {4,1,0,0}, {0,1,0,1}, },
  { {5,0,1,0}, {1,0,0,1}, {0,0,1,1}, {4,0,0,0}, },
};

struct cell_s IceIc = {
  8,
  nbrsIc,
  2.767,2.767,2.767*1.1,
  {2.3094010767585,2.3094010767585,2.3094010767585},
  0,
  {
    {0,   0   ,0   },
    {0.5 ,0.5 ,0   },
    {0.25,0.25,0.25},
    {0.75,0.75,0.25},
    {0   ,0.5 ,0.5 },
    {0.5 ,0   ,0.5},
    {0.75,0.25,0.75},
    {0.25,0.75,0.75}
  }
};

/******************* III *******************/
nbrs_t nbrsIII[12]={
  { {4,0,-1,0}, {3,0,0,0}, {9,0,0,-1}, {1,-1,0,0}, },
  { {0,1,0,0}, {9,0,0,-1}, {5,0,-1,0}, {2,0,0,0}, },
  { {1,0,0,0}, {5,0,0,0}, {11,0,0,-1}, {10,1,0,-1}, },
  { {8,0,0,0}, {4,0,0,0}, {7,0,0,0}, {0,0,0,0}, },
  { {0,0,1,0}, {5,0,0,0}, {3,0,0,0}, {6,0,1,0}, },
  { {6,1,1,0}, {1,0,1,0}, {4,0,0,0}, {2,0,0,0}, },
  { {7,0,0,0}, {5,-1,-1,0}, {4,0,-1,0}, {10,0,-1,0}, },
  { {3,0,0,0}, {8,-1,0,0}, {10,0,0,0}, {6,0,0,0}, },
  { {9,0,0,0}, {7,1,0,0}, {11,0,0,0}, {3,0,0,0}, },
  { {11,0,-1,0}, {0,0,0,1}, {8,0,0,0}, {1,0,0,1}, },
  { {2,-1,0,1}, {7,0,0,0}, {11,0,0,0}, {6,0,1,0}, },
  { {8,0,0,0}, {10,0,0,0}, {2,0,0,1}, {9,0,1,0}, },
};

struct cell_s IceIII = {
  12,
  nbrsIII,
  2.63,2.63,2.63*1.1,
  {2.5346,2.5346,2.6373},
  0,
  {
    {0.188, 0.188, 0. },
    {0.787, 0.188, 0. },
    {0.787, 0.588, 0. },
    {0.287, 0.500, 0.25 },
    {0.287, 0.900, 0.25 },
    {0.687, 0.900, 0.25 },
    {0.000, 0.000, 0.5 },
    {0.000, 0.400, 0.5 },
    {0.600, 0.400, 0.5 },
    {0.500, 0.087, 0.75 },
    {0.094, 0.687, 0.75 },
    {0.494, 0.687, 0.75 }
  }
};

#define SHIFT 5.5

struct cell_s IceV = {
  28,
  NULL,
  -1,2.7,3,
  {9.22,7.54,9.7742954}, /* Lz=c*sin(beta), c=10.35 */
  -3.4037698/9.22,      /* zshift=c*cos(beta)/a - see the code */
  { /* obtained using iceV.c */
    {2.40026,-1.05862,-4.74444+SHIFT},
    {-0.70585,-3.34399,3.378+SHIFT},
    {3.82107,2.37736,-4.88715+SHIFT},
    {0.81023,0.42601,-1.50915+SHIFT},
    {2.82464,-2.69555,-2.42109+SHIFT},
    {0.88418,2.71138,0.1427+SHIFT},
    {1.30857,1.07445,2.46605+SHIFT},
    {8.43108,-2.37736,-4.88715+SHIFT},
    {7.91143,-1.07445,-2.46605+SHIFT},
    {5.91857,-1.07445,2.46605+SHIFT},
    {5.42023,-0.42601,-1.50915+SHIFT},
    {7.43464,2.69555,-2.42109+SHIFT},
    {5.49418,-2.71138,0.1427+SHIFT},
    {5.31585,-3.34399,-3.378+SHIFT},
    {7.01026,1.05862,-4.74444+SHIFT},
    {5.24189,-1.05862,-5.02985+SHIFT},
    {9.92585,3.34399,-3.378+SHIFT},
    {9.85189,1.05862,-5.02985+SHIFT},
    {6.915,1.39264,0+SHIFT},
    {8.40977,-0.42601,1.50915+SHIFT},
    {6.39536,2.69555,2.42109+SHIFT},
    {8.33582,-2.71138,-0.1427+SHIFT},
    {3.90415,3.34399,3.378+SHIFT},
    {2.305,-1.39264,0+SHIFT},
    {3.79977,0.42601,1.50915+SHIFT},
    {1.78536,-2.69555,2.42109+SHIFT},
    {3.72582,2.71138,-0.1427+SHIFT},
    {3.30143,1.07445,-2.46605+SHIFT}
  }
};

nbrs_t nbrsVI[10]={
  { {6,0,0,0}, {8,0,0,0}, {2,0,0,1}, {4,0,0,1}, },
  { {7,0,0,0}, {9,0,0,0}, {3,0,0,-1}, {5,0,0,-1}, },
  { {6,0,0,0}, {8,0,0,0}, {0,0,0,-1}, {4,0,1,0}, },
  { {7,0,0,0}, {9,0,0,0}, {1,0,0,1}, {5,0,-1,0}, },
  { {6,0,0,0}, {8,0,0,0}, {0,0,0,-1}, {2,0,-1,0}, },
  { {7,0,0,0}, {9,0,0,0}, {1,0,0,1}, {3,0,1,0}, },
  { {0,0,0,0}, {2,0,0,0}, {4,0,0,0}, {8,-1,0,0}, },
  { {1,0,0,0}, {3,0,0,0}, {5,0,0,0}, {9,1,0,0}, },
  { {0,0,0,0}, {2,0,0,0}, {4,0,0,0}, {6,1,0,0}, },
  { {1,0,0,0}, {3,0,0,0}, {5,0,0,0}, {7,-1,0,0}, },
};

struct cell_s IceVI = {
  10,
  nbrsVI,
  2.7,2.7,3,
  {2.2893,2.2893,2.1104},
  0,
  {
    {.75,.25,.75},  // O1
    {.25,.75,.25},  // O1

    {.75,.53,.134}, // O2
    {.25,.47,.866}, // O2

    {.75,-.03,.134}, // O2
    {.25,1.03,.866}, // O2

    {.47,.25,.366}, // O2
    {.53,.75,.634}, // O2

    {1.03,.25,.366}, // O2
    {-.03,.75,.634} // O2
  }
};

nbrs_t nbrsVII[16]={
  { {2,0,0,0}, {3,-1,-1,0}, {6,-1,0,-1}, {7,0,-1,-1}, },
  { {2,0,0,0}, {3,0,0,0}, {6,0,0,-1}, {7,0,0,-1}, },
  { {0,0,0,0}, {1,0,0,0}, {4,0,0,0}, {5,0,0,0}, },
  { {0,1,1,0}, {1,0,0,0}, {4,1,0,0}, {5,0,1,0}, },
  { {2,0,0,0}, {3,-1,0,0}, {6,-1,0,0}, {7,0,0,0}, },
  { {2,0,0,0}, {3,0,-1,0}, {6,0,0,0}, {7,0,-1,0}, },
  { {0,1,0,1}, {1,0,0,1}, {4,1,0,0}, {5,0,0,0}, },
  { {0,0,1,1}, {1,0,0,1}, {4,0,0,0}, {5,0,1,0}, },
  { {10,0,0,0}, {11,0,-1,0}, {14,0,0,-1}, {15,0,-1,-1}, },
  { {10,-1,0,0}, {11,0,0,0}, {14,0,0,-1}, {15,-1,0,-1}, },
  { {8,0,0,0}, {9,1,0,0}, {12,0,0,0}, {13,1,0,0}, },
  { {8,0,1,0}, {9,0,0,0}, {12,0,0,0}, {13,0,1,0}, },
  { {10,0,0,0}, {11,0,0,0}, {14,0,0,0}, {15,0,0,0}, },
  { {10,-1,0,0}, {11,0,-1,0}, {14,0,0,0}, {15,-1,-1,0}, },
  { {8,0,0,1}, {9,0,0,1}, {12,0,0,0}, {13,0,0,0}, },
  { {8,0,1,1}, {9,1,0,1}, {12,0,0,0}, {13,1,1,0}, },
};

struct cell_s IceVII = {
  16,
  nbrsVII,
  2.861,2.861,2.861*1.1,
  {2.3094010767585,2.3094010767585,2.3094010767585},
  0,
  {
    {0,   0   ,0   },
    {0.5 ,0.5 ,0   },
    {0.25,0.25,0.25},
    {0.75,0.75,0.25},
    {0   ,0.5 ,0.5 },
    {0.5 ,0   ,0.5},
    {0.75,0.25,0.75},
    {0.25,0.75,0.75},

    {0.5 ,0   ,0   },
    {0   ,0.5 ,0   },
    {0.75,0.25,0.25},
    {0.25,0.75,0.25},
    {0.5, 0.5 ,0.5 },
    {0   ,0   ,0.5},
    {0.25,0.25,0.75},
    {0.75,0.75,0.75},
  }
};

/**************** clathrate I ***************/
nbrs_t nbrsClI[46]={
  { {1,0,0,0}, {17,1,0,0}, {4,0,0,0}, {13,0,0,0}, },
  { {42,0,0,0}, {2,0,0,0}, {16,0,-1,0}, {0,0,0,0}, },
  { {15,0,-1,0}, {3,0,0,0}, {1,0,0,0}, {41,0,0,0}, },
  { {9,0,0,0}, {4,0,0,0}, {21,1,0,0}, {2,0,0,0}, },
  { {12,0,0,0}, {0,0,0,0}, {3,0,0,0}, {23,1,0,0}, },
  { {9,0,0,0}, {31,0,0,0}, {12,0,0,0}, {6,0,0,0}, },
  { {15,0,0,0}, {5,0,0,0}, {11,0,0,0}, {44,0,0,0}, },
  { {35,0,0,0}, {13,0,0,0}, {12,0,0,0}, {8,0,0,0}, },
  { {16,0,0,0}, {7,0,0,0}, {11,0,0,1}, {45,0,0,0}, },
  { {10,0,0,0}, {40,0,0,0}, {5,0,0,0}, {3,0,0,0}, },
  { {13,0,0,-1}, {11,0,0,0}, {25,1,0,0}, {9,0,0,0}, },
  { {8,0,0,-1}, {6,0,0,0}, {10,0,0,0}, {26,1,0,0}, },
  { {5,0,0,0}, {7,0,0,0}, {14,0,0,0}, {4,0,0,0}, },
  { {0,0,0,0}, {43,0,0,0}, {10,0,0,1}, {7,0,0,0}, },
  { {12,0,0,0}, {16,0,0,0}, {15,0,0,0}, {33,1,0,0}, },
  { {14,0,0,0}, {2,0,1,0}, {38,1,0,0}, {6,0,0,0}, },
  { {39,1,0,0}, {1,0,1,0}, {14,0,0,0}, {8,0,0,0}, },
  { {18,0,0,0}, {39,0,-1,0}, {0,-1,0,0}, {25,0,0,1}, },
  { {30,0,0,0}, {19,0,0,0}, {17,0,0,0}, {23,0,0,0}, },
  { {20,0,0,0}, {42,0,0,0}, {34,0,-1,0}, {18,0,0,0}, },
  { {22,0,0,0}, {37,0,-1,0}, {19,0,0,0}, {41,0,0,0}, },
  { {22,0,0,0}, {3,-1,0,0}, {38,0,-1,0}, {25,0,0,0}, },
  { {27,0,0,0}, {23,0,0,0}, {21,0,0,0}, {20,0,0,0}, },
  { {18,0,0,0}, {4,-1,0,0}, {24,0,0,0}, {22,0,0,0}, },
  { {33,0,0,0}, {23,0,0,0}, {35,0,0,0}, {31,0,0,0}, },
  { {21,0,0,0}, {28,0,0,0}, {17,0,0,-1}, {10,-1,0,0}, },
  { {38,0,0,0}, {11,-1,0,0}, {29,0,0,0}, {39,0,0,-1}, },
  { {31,0,0,0}, {40,0,0,0}, {22,0,0,0}, {28,0,0,0}, },
  { {27,0,0,0}, {30,0,0,-1}, {25,0,0,0}, {29,0,0,0}, },
  { {26,0,0,0}, {28,0,0,0}, {32,0,0,0}, {36,0,0,-1}, },
  { {28,0,0,1}, {35,0,0,0}, {18,0,0,0}, {43,0,0,0}, },
  { {24,0,0,0}, {32,0,0,0}, {27,0,0,0}, {5,0,0,0}, },
  { {44,0,0,0}, {29,0,0,0}, {31,0,0,0}, {37,0,0,0}, },
  { {37,0,0,0}, {14,-1,0,0}, {24,0,0,0}, {34,0,0,0}, },
  { {33,0,0,0}, {39,0,0,0}, {36,0,0,0}, {19,0,1,0}, },
  { {36,0,0,0}, {24,0,0,0}, {7,0,0,0}, {30,0,0,0}, },
  { {34,0,0,0}, {45,0,0,0}, {29,0,0,1}, {35,0,0,0}, },
  { {20,0,1,0}, {32,0,0,0}, {33,0,0,0}, {38,0,0,0}, },
  { {21,0,1,0}, {37,0,0,0}, {26,0,0,0}, {15,-1,0,0}, },
  { {16,-1,0,0}, {17,0,1,0}, {34,0,0,0}, {26,0,0,1}, },
  { {43,0,0,-1}, {41,0,0,0}, {27,0,0,0}, {9,0,0,0}, },
  { {2,0,0,0}, {20,0,0,0}, {40,0,0,0}, {44,0,-1,0}, },
  { {43,0,0,0}, {45,0,-1,0}, {19,0,0,0}, {1,0,0,0}, },
  { {40,0,0,1}, {30,0,0,0}, {13,0,0,0}, {42,0,0,0}, },
  { {6,0,0,0}, {45,0,0,-1}, {32,0,0,0}, {41,0,1,0}, },
  { {42,0,1,0}, {8,0,0,0}, {36,0,0,0}, {44,0,0,1}, },
};

struct cell_s ClI = {
  46,
  nbrsClI,
  2.767,2.767,2.767*1.1,
  {4.3,4.3,4.3},
  0,
  {
    {0.81987442, 0.18612558, 0.68172791},
    {0.7024814, -0.0017813953, 0.61632093},
    {0.70211395, -0.0011395349, 0.38031395},
    {0.81930233, 0.18712326, 0.31556744},
    {0.88065349, 0.30478372, 0.49887442},
    {0.62060698, 0.50096047, 0.3008907},
    {0.68439535, 0.68691395, 0.18278605},
    {0.62122791, 0.49987674, 0.69873488},
    {0.68538837, 0.68518372, 0.81765116},
    {0.68567674, 0.3161, 0.18177442},
    {0.80477674, 0.38250698, -0.0055697674},
    {0.80394884, 0.62235349, -0.0049139535},
    {0.75004651, 0.50086512, 0.49961163},
    {0.68666977, 0.31436977, 0.81663721},
    {0.87929535, 0.69784651, 0.49994651},
    {0.81713023, 0.81607907, 0.31728372},
    {0.81770233, 0.8150814, 0.68344419},
    {-0.0035860465, 0.12393256, 0.79922558},
    {0.18286977, 0.18392093, 0.68271628},
    {0.30135581, -0.0031697674, 0.61694419},
    {0.30098837, -0.0025255814, 0.38093721},
    {-0.0045209302, 0.12556512, 0.20030465},
    {0.18229767, 0.1849186, 0.31655581},
    {0.12070465, 0.30215349, 0.50005349},
    {0.24995349, 0.49913488, 0.50038837},
    {-0.0058906977, 0.25617907, -0.0015023256},
    {-0.0075744186, 0.74305349, -0.00017209302},
    {0.31461163, 0.31481628, 0.18234884},
    {0.19447442, 0.38039535, -0.0046209302},
    {0.19364651, 0.62024419, -0.0039674419},
    {0.31560465, 0.31308605, 0.81721395},
    {0.37877209, 0.50012326, 0.30126512},
    {0.31333023, 0.68563023, 0.18336279},
    {0.11934651, 0.69521628, 0.50112558},
    {0.18069767, 0.81287674, 0.68443256},
    {0.37939302, 0.49903953, 0.6991093},
    {0.31432326, 0.6839, 0.81822558},
    {0.18012558, 0.81387442, 0.31827209},
    {-0.0071023256, 0.87256977, 0.20234419},
    {-0.0061651163, 0.87093953, 0.80126512},
    {0.50048605, 0.1917907, 0.12602326},
    {0.50135116, 0.0039674419, 0.26518837},
    {0.5020814, 0.0026953488, 0.7321},
    {0.50165116, 0.18975814, 0.87228837},
    {0.49834884, 0.81024186, 0.12771163},
    {0.49951395, 0.8082093, 0.87397674},
  }
};

/**************** clathrate II ***************/
nbrs_t nbrsClII[136]={
{ {135,0,0,0}, {3,0,0,0}, {2,0,0,0}, {1,0,0,0}, },
{ {6,0,0,0}, {0,0,0,0}, {81,0,0,0}, {4,0,0,0}, },
{ {5,0,0,0}, {0,0,0,0}, {23,0,0,0}, {105,0,0,0}, },
{ {46,-1,0,0}, {29,0,0,0}, {0,0,0,0}, {94,-1,0,0}, },
{ {40,0,0,0}, {1,0,0,0}, {5,0,0,0}, {44,0,0,0}, },
{ {43,0,0,0}, {4,0,0,0}, {2,0,0,0}, {7,0,0,0}, },
{ {13,0,0,0}, {14,0,0,0}, {46,-1,0,0}, {1,0,0,0}, },
{ {5,0,0,0}, {69,0,0,0}, {8,0,0,0}, {42,0,0,0}, },
{ {60,-1,0,0}, {7,0,0,0}, {12,0,0,0}, {21,0,0,0}, },
{ {59,-1,0,0}, {20,0,0,1}, {44,0,0,0}, {13,0,0,0}, },
{ {11,0,0,0}, {78,-1,1,0}, {100,-1,0,1}, {71,0,0,0}, },
{ {42,0,0,0}, {60,-1,0,0}, {10,0,0,0}, {59,-1,0,0}, },
{ {8,0,0,0}, {31,0,1,0}, {15,0,1,0}, {85,0,1,0}, },
{ {49,-1,0,0}, {9,0,0,0}, {6,0,0,0}, {16,0,0,0}, },
{ {17,0,0,0}, {15,0,0,0}, {6,0,0,0}, {84,0,0,0}, },
{ {12,0,-1,0}, {57,-1,-1,0}, {73,-1,0,0}, {14,0,0,0}, },
{ {13,0,0,0}, {18,0,0,1}, {17,0,0,0}, {34,-1,0,0}, },
{ {16,0,0,0}, {78,-1,0,0}, {89,0,0,0}, {14,0,0,0}, },
{ {90,-1,0,0}, {27,0,0,0}, {16,0,0,-1}, {20,0,0,0}, },
{ {23,0,0,0}, {25,0,0,0}, {108,0,0,0}, {22,0,0,0}, },
{ {9,0,0,-1}, {18,0,0,0}, {22,0,0,0}, {107,0,0,0}, },
{ {23,0,0,0}, {8,0,0,0}, {24,0,0,0}, {93,-1,0,0}, },
{ {20,0,0,0}, {19,0,0,0}, {95,-1,0,0}, {102,-1,0,0}, },
{ {94,-1,0,0}, {2,0,0,0}, {19,0,0,0}, {21,0,0,0}, },
{ {21,0,0,0}, {120,-1,0,0}, {31,0,1,0}, {25,0,0,0}, },
{ {100,-1,0,0}, {24,0,0,0}, {19,0,0,0}, {26,0,0,0}, },
{ {71,0,0,-1}, {25,0,0,0}, {33,0,1,0}, {113,0,0,0}, },
{ {28,0,0,0}, {104,0,0,0}, {30,0,0,0}, {18,0,0,0}, },
{ {129,0,0,0}, {33,0,0,0}, {89,0,0,-1}, {27,0,0,0}, },
{ {30,0,0,0}, {90,-1,0,0}, {3,0,0,0}, {128,-1,0,0}, },
{ {32,0,0,0}, {27,0,0,0}, {29,0,0,0}, {135,0,0,0}, },
{ {32,0,0,0}, {128,-1,0,0}, {12,0,-1,0}, {24,0,-1,0}, },
{ {33,0,0,0}, {82,0,0,0}, {31,0,0,0}, {30,0,0,0}, },
{ {124,0,-1,0}, {26,0,-1,0}, {28,0,0,0}, {32,0,0,0}, },
{ {76,0,0,0}, {91,0,0,1}, {35,0,0,0}, {16,1,0,0}, },
{ {66,0,0,0}, {34,0,0,0}, {49,0,0,0}, {36,0,0,0}, },
{ {35,0,0,0}, {37,0,0,0}, {75,0,0,0}, {38,0,0,0}, },
{ {83,0,0,0}, {47,0,0,0}, {36,0,0,0}, {88,0,0,0}, },
{ {36,0,0,0}, {64,0,0,0}, {39,0,0,0}, {87,0,0,0}, },
{ {38,0,0,0}, {81,0,0,0}, {40,0,0,0}, {83,0,0,0}, },
{ {45,0,0,0}, {39,0,0,0}, {65,0,0,0}, {4,0,0,0}, },
{ {110,0,0,0}, {83,0,0,0}, {45,0,0,0}, {48,0,0,0}, },
{ {44,0,0,0}, {7,0,0,0}, {11,0,0,0}, {72,0,0,0}, },
{ {109,0,0,0}, {45,0,0,0}, {70,0,0,0}, {5,0,0,0}, },
{ {4,0,0,0}, {9,0,0,0}, {42,0,0,0}, {63,0,0,0}, },
{ {41,0,0,0}, {56,0,0,0}, {40,0,0,0}, {43,0,0,0}, },
{ {73,0,0,0}, {47,0,0,0}, {6,1,0,0}, {3,1,0,0}, },
{ {49,0,0,0}, {48,0,0,0}, {37,0,0,0}, {46,0,0,0}, },
{ {41,0,0,0}, {62,0,0,0}, {47,0,0,0}, {92,0,0,0}, },
{ {35,0,0,0}, {13,1,0,0}, {61,0,0,0}, {47,0,0,0}, },
{ {52,0,0,0}, {51,0,0,0}, {67,0,0,0}, {97,0,0,1}, },
{ {74,0,1,0}, {68,0,0,0}, {54,0,0,0}, {50,0,0,0}, },
{ {56,0,0,0}, {65,0,0,0}, {50,0,0,0}, {53,0,0,0}, },
{ {52,0,0,0}, {101,0,0,1}, {66,0,0,0}, {61,0,0,0}, },
{ {70,0,0,0}, {51,0,0,0}, {56,0,0,0}, {55,0,0,0}, },
{ {77,0,1,0}, {54,0,0,0}, {122,0,0,0}, {58,0,0,0}, },
{ {62,0,0,0}, {54,0,0,0}, {52,0,0,0}, {45,0,0,0}, },
{ {15,1,1,0}, {78,0,1,0}, {60,0,0,0}, {77,0,1,0}, },
{ {55,0,0,0}, {93,0,0,0}, {62,0,0,0}, {60,0,0,0}, },
{ {102,0,0,1}, {61,0,0,0}, {9,1,0,0}, {11,1,0,0}, },
{ {57,0,0,0}, {58,0,0,0}, {11,1,0,0}, {8,1,0,0}, },
{ {53,0,0,0}, {49,0,0,0}, {62,0,0,0}, {59,0,0,0}, },
{ {58,0,0,0}, {61,0,0,0}, {56,0,0,0}, {48,0,0,0}, },
{ {44,0,0,0}, {67,0,0,0}, {65,0,0,0}, {107,0,0,1}, },
{ {112,0,0,1}, {65,0,0,0}, {66,0,0,0}, {38,0,0,0}, },
{ {63,0,0,0}, {40,0,0,0}, {52,0,0,0}, {64,0,0,0}, },
{ {53,0,0,0}, {64,0,0,0}, {116,0,0,1}, {35,0,0,0}, },
{ {72,0,0,0}, {50,0,0,0}, {113,0,0,1}, {63,0,0,0}, },
{ {72,0,0,0}, {69,0,0,0}, {51,0,0,0}, {86,0,1,0}, },
{ {70,0,0,0}, {85,0,1,0}, {7,0,0,0}, {68,0,0,0}, },
{ {43,0,0,0}, {123,0,0,0}, {54,0,0,0}, {69,0,0,0}, },
{ {10,0,0,0}, {89,0,1,0}, {26,0,0,1}, {72,0,0,0}, },
{ {71,0,0,0}, {42,0,0,0}, {68,0,0,0}, {67,0,0,0}, },
{ {88,0,0,0}, {128,0,0,0}, {15,1,0,0}, {46,0,0,0}, },
{ {87,0,0,0}, {51,0,-1,0}, {96,0,-1,1}, {75,0,0,0}, },
{ {36,0,0,0}, {74,0,0,0}, {77,0,0,0}, {76,0,0,0}, },
{ {75,0,0,0}, {78,0,0,0}, {34,0,0,0}, {125,0,0,1}, },
{ {75,0,0,0}, {88,0,0,0}, {55,0,-1,0}, {57,0,-1,0}, },
{ {10,1,-1,0}, {17,1,0,0}, {57,0,-1,0}, {76,0,0,0}, },
{ {81,0,0,0}, {135,0,0,0}, {80,0,0,0}, {82,0,0,0}, },
{ {79,0,0,0}, {131,0,0,0}, {83,0,0,0}, {106,0,0,0}, },
{ {84,0,0,0}, {1,0,0,0}, {39,0,0,0}, {79,0,0,0}, },
{ {85,0,0,0}, {79,0,0,0}, {32,0,0,0}, {134,0,0,0}, },
{ {39,0,0,0}, {80,0,0,0}, {37,0,0,0}, {41,0,0,0}, },
{ {86,0,0,0}, {14,0,0,0}, {81,0,0,0}, {85,0,0,0}, },
{ {69,0,-1,0}, {84,0,0,0}, {82,0,0,0}, {12,0,-1,0}, },
{ {87,0,0,0}, {89,0,0,0}, {68,0,-1,0}, {84,0,0,0}, },
{ {38,0,0,0}, {129,0,0,1}, {74,0,0,0}, {86,0,0,0}, },
{ {37,0,0,0}, {130,0,0,0}, {77,0,0,0}, {73,0,0,0}, },
{ {17,0,0,0}, {28,0,0,1}, {71,0,-1,0}, {86,0,0,0}, },
{ {91,0,0,0}, {18,1,0,0}, {29,1,0,0}, {95,0,0,0}, },
{ {126,0,0,0}, {116,0,0,0}, {34,0,0,-1}, {90,0,0,0}, },
{ {94,0,0,0}, {48,0,0,0}, {117,0,0,0}, {93,0,0,0}, },
{ {92,0,0,0}, {21,1,0,0}, {58,0,0,0}, {121,0,0,0}, },
{ {3,1,0,0}, {95,0,0,0}, {23,1,0,0}, {92,0,0,0}, },
{ {90,0,0,0}, {22,1,0,0}, {118,0,0,0}, {94,0,0,0}, },
{ {74,0,1,-1}, {133,0,1,0}, {97,0,0,0}, {125,0,1,0}, },
{ {96,0,0,0}, {115,0,0,0}, {50,0,0,-1}, {99,0,0,0}, },
{ {125,0,1,0}, {99,0,0,0}, {120,0,0,0}, {100,0,0,0}, },
{ {119,0,0,0}, {97,0,0,0}, {98,0,0,0}, {101,0,0,0}, },
{ {98,0,0,0}, {25,1,0,0}, {102,0,0,0}, {10,1,0,-1}, },
{ {99,0,0,0}, {102,0,0,0}, {53,0,0,-1}, {118,0,0,0}, },
{ {59,0,0,-1}, {100,0,0,0}, {101,0,0,0}, {22,1,0,0}, },
{ {106,0,0,0}, {105,0,0,0}, {104,0,0,0}, {135,0,0,0}, },
{ {107,0,0,0}, {103,0,0,0}, {27,0,0,0}, {112,0,0,0}, },
{ {108,0,0,0}, {2,0,0,0}, {103,0,0,0}, {109,0,0,0}, },
{ {110,0,0,0}, {80,0,0,0}, {111,0,0,0}, {103,0,0,0}, },
{ {108,0,0,0}, {20,0,0,0}, {104,0,0,0}, {63,0,0,-1}, },
{ {113,0,0,0}, {19,0,0,0}, {107,0,0,0}, {105,0,0,0}, },
{ {114,0,0,0}, {105,0,0,0}, {110,0,0,0}, {43,0,0,0}, },
{ {117,0,0,0}, {109,0,0,0}, {41,0,0,0}, {106,0,0,0}, },
{ {106,0,0,0}, {112,0,0,0}, {116,0,0,0}, {132,0,0,0}, },
{ {64,0,0,-1}, {104,0,0,0}, {111,0,0,0}, {129,0,0,0}, },
{ {67,0,0,-1}, {26,0,0,0}, {108,0,0,0}, {115,0,0,0}, },
{ {119,0,0,0}, {123,0,0,0}, {115,0,0,0}, {109,0,0,0}, },
{ {114,0,0,0}, {113,0,0,0}, {97,0,0,0}, {124,0,0,0}, },
{ {118,0,0,0}, {111,0,0,0}, {66,0,0,-1}, {91,0,0,0}, },
{ {118,0,0,0}, {92,0,0,0}, {119,0,0,0}, {110,0,0,0}, },
{ {101,0,0,0}, {95,0,0,0}, {116,0,0,0}, {117,0,0,0}, },
{ {117,0,0,0}, {121,0,0,0}, {114,0,0,0}, {99,0,0,0}, },
{ {127,0,1,0}, {98,0,0,0}, {121,0,0,0}, {24,1,0,0}, },
{ {93,0,0,0}, {120,0,0,0}, {119,0,0,0}, {122,0,0,0}, },
{ {55,0,0,0}, {121,0,0,0}, {130,0,1,0}, {123,0,0,0}, },
{ {134,0,1,0}, {122,0,0,0}, {114,0,0,0}, {70,0,0,0}, },
{ {115,0,0,0}, {134,0,1,0}, {33,0,1,0}, {133,0,1,0}, },
{ {126,0,0,0}, {76,0,0,-1}, {98,0,-1,0}, {96,0,-1,0}, },
{ {132,0,0,0}, {127,0,0,0}, {91,0,0,0}, {125,0,0,0}, },
{ {130,0,0,0}, {120,0,-1,0}, {128,0,0,0}, {126,0,0,0}, },
{ {127,0,0,0}, {31,1,0,0}, {73,0,0,0}, {29,1,0,0}, },
{ {87,0,0,-1}, {112,0,0,0}, {28,0,0,0}, {133,0,0,0}, },
{ {122,0,-1,0}, {131,0,0,0}, {127,0,0,0}, {88,0,0,0}, },
{ {134,0,0,0}, {132,0,0,0}, {130,0,0,0}, {80,0,0,0}, },
{ {133,0,0,0}, {111,0,0,0}, {126,0,0,0}, {131,0,0,0}, },
{ {96,0,-1,0}, {129,0,0,0}, {132,0,0,0}, {124,0,-1,0}, },
{ {82,0,0,0}, {124,0,-1,0}, {123,0,-1,0}, {131,0,0,0}, },
{ {103,0,0,0}, {30,0,0,0}, {79,0,0,0}, {0,0,0,0}, },
};

struct cell_s ClII = {
  136,
  nbrsClII,
  -1,2.767,2.767*1.1,
  {17.263305,17.263305,17.263305},
  0,
  {
    {  -2.95726,  1.14388,  1.73422},
    {  -2.38005,  0.57291,  4.38790},
    {  -2.38151,  3.79943,  1.14811},
    {  -5.61012,  0.56816,  1.15923},
    {  -1.34286,  2.92664,  5.44871},
    {  -1.29559,  4.90242,  3.46180},
    {  -4.70573, -0.51056,  5.47766},
    {  -2.37879,  7.22135,  4.55370},
    {  -4.70042,  8.30560,  3.46391},
    {  -4.70476,  2.91399,  8.88332},
    {  -5.61287,  9.22877,  9.77609},
    {  -5.60981,  7.22956,  7.78628},
    {  -4.12037, 10.95367,  2.88055},
    {  -5.79319,  0.58610,  7.79942},
    {  -4.13266, -3.16230,  6.06272},
    {  -5.70538, -4.74208,  4.49378},
    {  -5.79625, -1.40466,  9.79828},
    {  -4.71202, -3.73198,  8.71760},
    {  -4.70397, -0.35094, -5.14951},
    {  -4.12425,  5.45300, -2.59464},
    {  -4.12567,  2.29975, -5.73279},
    {  -5.78740,  7.21164,  1.14226},
    {  -5.69788,  3.87835, -4.16492},
    {  -4.70198,  4.88409,  0.05763},
    {  -5.78330,  9.19972, -0.85790},
    {  -4.69684,  8.10461, -3.17752},
    {  -2.37278,  9.18731, -4.26468},
    {  -2.38299, -1.43638, -4.06010},
    {  -1.29925, -3.75723, -5.15223},
    {  -5.61110, -1.43038, -0.82829},
    {  -2.95848, -2.00853, -1.40163},
    {  -4.70715, -5.74800,  0.27438},
    {  -2.38195, -4.65838, -0.81538},
    {  -1.34795, -5.73247, -3.16451},
    {   8.81648, -1.99512, 10.38860},
    {   7.24617, -0.42072,  8.81025},
    {   5.67356, -1.99740,  7.24192},
    {   6.24957, -1.42652,  4.58647},
    {   3.02226, -1.41688,  7.81603},
    {   1.93641, -0.33460,  5.49140},
    {   1.34055,  2.31112,  6.06860},
    {   4.49996,  2.31185,  2.91359},
    {  -2.95571,  6.64974,  7.21297},
    {   1.35918,  5.46270,  2.89894},
    {  -2.37817,  3.99835,  7.79708},
    {   2.91843,  3.88761,  4.48287},
    {  10.56568, -0.52800,  3.49547},
    {   8.24384,  0.56773,  4.58137},
    {   7.15492,  2.88823,  3.48817},
    {   8.81955,  1.15021,  7.23390},
    {   2.84686,  7.22652,  9.78600},
    {   2.84832,  9.21585,  7.78649},
    {   3.93186,  4.89747,  8.70413},
    {   6.25351,  3.81274,  9.79801},
    {   3.93415,  8.11842,  5.46733},
    {   6.25933,  9.20267,  4.37864},
    {   4.50278,  5.46569,  6.05009},
    {  10.00351, 10.95344,  6.03704},
    {   8.25077,  7.20690,  4.38504},
    {  10.57322,  4.89714,  8.89351},
    {  10.57753,  8.29888,  5.47281},
    {   8.24757,  3.80672,  7.80422},
    {   7.15754,  4.88705,  5.47717},
    {  -0.38197,  3.99490,  9.79073},
    {   3.02181,  0.58220,  9.80435},
    {   1.93791,  2.90113,  8.70996},
    {   5.67435,  1.16002, 10.37861},
    {   0.19288,  6.65165, 10.35885},
    {   0.19503,  9.79024,  7.20579},
    {  -0.38104,  9.21099,  4.55170},
    {   1.94055,  8.11572,  3.47000},
    {  -2.95818,  9.80008, 10.34859},
    {  -1.38108,  8.22284,  8.78111},
    {   9.98759, -3.18044,  2.92739},
    {   3.92590, -5.73240,  8.91797},
    {   6.24900, -4.64945,  7.82645},
    {   8.24178, -4.64990,  9.82194},
    {   7.33755, -5.74391,  5.50641},
    {  10.56705, -5.74261,  8.73638},
    {   0.19137, -2.00246,  1.74743},
    {   2.84172, -1.43145,  1.17193},
    {  -0.38381, -1.41907,  4.40035},
    {  -0.38300, -4.65347,  1.17632},
    {   3.92824, -0.33785,  3.49550},
    {  -1.47762, -3.73587,  5.49187},
    {  -1.47249, -5.72792,  3.49973},
    {  -0.39095, -4.82086,  7.82350},
    {   1.93250, -3.73483,  8.90935},
    {   7.33710, -3.75391,  3.50552},
    {  -2.39103, -4.81791,  9.81110},
    {  10.56781, -0.36254, -3.14092},
    {   8.24430, -1.45344, -4.22979},
    {   8.24614,  3.96655,  1.16277},
    {   8.82412,  6.62190,  1.73242},
    {  10.56865,  2.87200,  0.07725},
    {   9.99123,  2.29101, -2.57435},
    {   4.51932, 10.92474, -5.74997},
    {   3.94037,  8.27285, -5.16707},
    {   8.25740,  9.17790, -4.08200},
    {   6.26070,  7.18573, -4.07395},
    {  10.58272,  8.08405, -5.16671},
    {   7.34702,  4.85907, -5.15506},
    {  10.00063,  5.43114, -5.73166},
    {   0.18930,  1.13532, -1.40889},
    {  -0.38641,  0.55284, -4.06141},
    {  -0.38447,  3.78958, -0.84152},
    {   2.84131,  0.55742, -0.82811},
    {  -1.47179,  2.87344, -5.15584},
    {  -1.47131,  4.87276, -3.16726},
    {   1.93841,  4.87816,  0.24781},
    {   3.92966,  2.88107,  0.25602},
    {   3.92858, -0.54056, -3.14808},
    {   1.93568, -0.54320, -5.14418},
    {  -0.37968,  7.19118, -4.26004},
    {   3.02911,  7.19635, -0.84524},
    {   1.94394,  8.27917, -3.17274},
    {   6.25189,  0.54322, -4.23731},
    {   6.25348,  3.96408, -0.83428},
    {   7.34137,  2.86843, -3.15471},
    {   5.68134,  6.61605, -1.41908},
    {   8.82947,  9.75717, -1.42886},
    {   7.25447,  8.19147,  0.15228},
    {   5.68301,  9.76899,  1.72221},
    {   3.02936,  9.19548,  1.14431},
    {   1.36974, 10.93193, -2.60261},
    {   7.15484, -5.77244, -5.12625},
    {   7.15411, -3.77241, -3.13781},
    {   8.24321, -4.85298, -0.81104},
    {  10.56578, -3.76230,  0.27757},
    {   1.35529, -3.19712, -5.71414},
    {   6.25101, -4.84649,  1.18329},
    {   3.92882, -3.75984,  0.08964},
    {   4.49935, -3.19378, -2.56489},
    {   2.91571, -4.77180, -4.13089},
    {   1.93837, -5.75543,  0.09750},
    {  -1.38369, -0.43347,  0.16796},
  }
};

/**************** clathrate H ***************/
nbrs_t nbrsClH[68]={
{ {40,0,0,0}, {28,0,0,0}, {2,0,0,0}, {8,0,0,0}, },
{ {2,0,0,0}, {3,0,0,0}, {43,0,0,-1}, {5,0,0,0}, },
{ {12,0,1,0}, {0,0,0,0}, {9,0,1,0}, {1,0,0,0}, },
{ {6,0,1,0}, {4,0,0,0}, {11,0,1,0}, {1,0,0,0}, },
{ {44,0,0,0}, {17,0,0,0}, {3,0,0,0}, {34,-1,0,0}, },
{ {1,0,0,0}, {8,0,0,0}, {17,0,0,0}, {51,0,0,-1}, },
{ {49,0,0,-1}, {19,0,0,0}, {3,0,-1,0}, {21,-1,-1,0}, },
{ {48,0,0,-1}, {13,0,0,0}, {8,0,0,0}, {22,0,0,0}, },
{ {0,0,0,0}, {7,0,0,0}, {10,0,0,0}, {5,0,0,0}, },
{ {2,0,-1,0}, {24,0,-1,0}, {50,0,0,-1}, {13,0,0,0}, },
{ {8,0,0,0}, {14,0,0,0}, {45,0,0,0}, {15,0,0,0}, },
{ {12,0,0,0}, {3,0,-1,0}, {16,0,0,0}, {41,0,-1,0}, },
{ {2,0,-1,0}, {42,0,-1,0}, {11,0,0,0}, {14,0,0,0}, },
{ {14,0,0,0}, {9,0,0,0}, {7,0,0,0}, {39,0,0,0}, },
{ {47,0,0,0}, {12,0,0,0}, {10,0,0,0}, {13,0,0,0}, },
{ {52,0,0,0}, {10,0,0,0}, {16,0,0,0}, {17,0,0,0}, },
{ {11,0,0,0}, {15,0,0,0}, {46,0,0,0}, {19,0,0,0}, },
{ {15,0,0,0}, {5,0,0,0}, {4,0,0,0}, {18,0,0,0}, },
{ {17,0,0,0}, {33,-1,0,0}, {19,0,0,0}, {53,0,0,-1}, },
{ {18,0,0,0}, {16,0,0,0}, {35,-1,0,0}, {6,0,0,0}, },
{ {60,0,0,-1}, {36,0,0,0}, {23,0,0,0}, {32,0,0,0}, },
{ {58,0,0,-1}, {34,0,0,0}, {38,0,1,0}, {6,1,1,0}, },
{ {57,0,0,-1}, {7,0,0,0}, {28,0,0,0}, {23,0,0,0}, },
{ {20,0,0,0}, {22,0,0,0}, {39,0,0,0}, {25,0,0,0}, },
{ {28,0,0,0}, {59,0,0,-1}, {37,0,1,0}, {9,0,1,0}, },
{ {30,0,0,0}, {23,0,0,0}, {29,0,0,0}, {54,0,0,0}, },
{ {64,0,1,0}, {31,0,0,0}, {27,0,0,0}, {38,0,1,0}, },
{ {37,0,1,0}, {26,0,0,0}, {29,0,0,0}, {65,0,1,0}, },
{ {29,0,0,0}, {22,0,0,0}, {24,0,0,0}, {0,0,0,0}, },
{ {25,0,0,0}, {27,0,0,0}, {28,0,0,0}, {56,0,0,0}, },
{ {61,0,0,0}, {32,0,0,0}, {25,0,0,0}, {31,0,0,0}, },
{ {30,0,0,0}, {34,0,0,0}, {55,0,0,0}, {26,0,0,0}, },
{ {35,0,0,0}, {20,0,0,0}, {30,0,0,0}, {33,0,0,0}, },
{ {62,0,0,-1}, {32,0,0,0}, {18,1,0,0}, {34,0,0,0}, },
{ {33,0,0,0}, {4,1,0,0}, {21,0,0,0}, {31,0,0,0}, },
{ {19,1,0,0}, {63,0,0,0}, {32,0,0,0}, {38,0,0,0}, },
{ {66,0,0,-1}, {37,0,0,0}, {38,0,0,0}, {20,0,0,0}, },
{ {39,0,0,0}, {27,0,-1,0}, {24,0,-1,0}, {36,0,0,0}, },
{ {36,0,0,0}, {35,0,0,0}, {26,0,-1,0}, {21,0,-1,0}, },
{ {23,0,0,0}, {13,0,0,0}, {67,0,0,0}, {37,0,0,0}, },
{ {42,0,0,0}, {45,0,0,0}, {0,0,0,0}, {56,0,0,0}, },
{ {44,0,0,0}, {43,0,0,0}, {49,0,1,0}, {11,0,1,0}, },
{ {50,0,1,0}, {43,0,0,0}, {40,0,0,0}, {12,0,1,0}, },
{ {51,0,0,0}, {1,0,0,1}, {42,0,0,0}, {41,0,0,0}, },
{ {52,0,0,0}, {55,-1,0,0}, {4,0,0,0}, {41,0,0,0}, },
{ {48,0,0,0}, {10,0,0,0}, {40,0,0,0}, {51,0,0,0}, },
{ {49,0,0,0}, {16,0,0,0}, {63,-1,0,0}, {53,0,0,0}, },
{ {67,0,0,0}, {48,0,0,0}, {14,0,0,0}, {50,0,0,0}, },
{ {57,0,0,0}, {7,0,0,1}, {47,0,0,0}, {45,0,0,0}, },
{ {6,0,0,1}, {41,0,-1,0}, {46,0,0,0}, {58,-1,-1,0}, },
{ {9,0,0,1}, {47,0,0,0}, {59,0,-1,0}, {42,0,-1,0}, },
{ {45,0,0,0}, {5,0,0,1}, {43,0,0,0}, {52,0,0,0}, },
{ {51,0,0,0}, {53,0,0,0}, {44,0,0,0}, {15,0,0,0}, },
{ {62,-1,0,0}, {46,0,0,0}, {18,0,0,1}, {52,0,0,0}, },
{ {67,0,0,0}, {25,0,0,0}, {60,0,0,0}, {57,0,0,0}, },
{ {44,1,0,0}, {31,0,0,0}, {62,0,0,0}, {58,0,0,0}, },
{ {40,0,0,0}, {29,0,0,0}, {59,0,0,0}, {57,0,0,0}, },
{ {56,0,0,0}, {54,0,0,0}, {48,0,0,0}, {22,0,0,1}, },
{ {55,0,0,0}, {64,0,1,0}, {21,0,0,1}, {49,1,1,0}, },
{ {56,0,0,0}, {65,0,1,0}, {24,0,0,1}, {50,0,1,0}, },
{ {61,0,0,0}, {54,0,0,0}, {66,0,0,0}, {20,0,0,1}, },
{ {62,0,0,0}, {63,0,0,0}, {60,0,0,0}, {30,0,0,0}, },
{ {33,0,0,1}, {55,0,0,0}, {61,0,0,0}, {53,1,0,0}, },
{ {46,1,0,0}, {64,0,0,0}, {61,0,0,0}, {35,0,0,0}, },
{ {26,0,-1,0}, {66,0,0,0}, {63,0,0,0}, {58,0,-1,0}, },
{ {59,0,-1,0}, {27,0,-1,0}, {67,0,0,0}, {66,0,0,0}, },
{ {65,0,0,0}, {60,0,0,0}, {64,0,0,0}, {36,0,0,1}, },
{ {65,0,0,0}, {39,0,0,0}, {47,0,0,0}, {54,0,0,0}, },
};

struct cell_s ClH = {
  68,
  nbrsClH,
  -1,2.767,2.767*1.1,
  {21.245, 12.385, 10.023},
  0,
  {
    {  -1.31262,  4.10950, -1.36087},
    {  -4.86481,  5.52898, -3.62065},
    {  -2.60858,  6.37015, -2.25148},
    {  -7.11536,  6.38860, -2.25148},
    {  -8.42587,  4.13893, -1.36053},
    {  -4.87434,  2.72049, -3.62118},
    {  -9.01889, -4.38696, -3.62160},
    {  -0.76061,  0.33023, -3.62206},
    {  -2.62275,  1.86068, -2.25258},
    {  -0.77161, -4.41779, -3.62331},
    {  -3.46802,  0.40873,  0.00002},
    {  -6.31023, -4.46271, -0.00162},
    {  -3.48046, -4.47390, -0.00108},
    {  -0.37651, -2.04587, -2.25323},
    {  -2.05974, -2.03805, -0.00083},
    {  -6.29375,  0.42014,  0.00029},
    {  -7.71561, -2.01650, -0.00024},
    {  -7.13049,  1.87932, -2.25276},
    {  -9.00118,  0.36213, -3.62202},
    {  -9.39886, -2.01153, -2.25312},
    {   5.79043, -0.66665, -3.62239},
    {   9.93459,  6.44064, -3.62001},
    {   1.67670,  1.72361, -3.62151},
    {   3.53875,  0.19218, -2.25351},
    {   1.68770,  6.47163, -3.62026},
    {   4.38412,  1.64280, -0.00013},
    {   7.22670,  6.51530,  0.00064},
    {   4.39674,  6.52641,  0.00108},
    {   1.29231,  4.09866, -2.25188},
    {   2.97602,  4.09056,  0.00083},
    {   7.20986,  1.63139, -0.00040},
    {   8.63188,  4.06901,  0.00024},
    {   8.04649,  0.17354, -2.25333},
    {   9.91745,  1.69179, -3.62254},
    {  10.31466,  4.06431, -2.25199},
    {   9.34165, -2.08594, -1.36279},
    {   5.78109, -3.47506, -3.62391},
    {   3.52554, -4.31747, -2.25444},
    {   8.03154, -4.33566, -2.25559},
    {   2.22937, -2.05670, -1.36228},
    {  -1.31310,  4.10921,  1.36228},
    {  -7.11527,  6.38817,  2.25559},
    {  -2.60847,  6.36881,  2.25450},
    {  -4.86481,  5.52757,  3.62391},
    {  -8.42554,  4.13747,  1.36268},
    {  -2.62264,  1.85934,  2.25339},
    {  -9.39855, -2.01278,  2.25188},
    {  -0.37620, -2.04713,  2.25177},
    {  -0.76059,  0.32792,  3.62140},
    {  -9.01849, -4.38911,  3.61990},
    {  -0.77142, -4.41912,  3.62026},
    {  -4.87319,  2.71897,  3.62256},
    {  -7.13038,  1.87799,  2.25322},
    {  -9.00038,  0.35955,  3.62260},
    {   3.53886,  0.19085,  2.25247},
    {  10.31516,  4.06313,  2.25203},
    {   1.29261,  4.09740,  2.25312},
    {   1.67689,  1.72228,  3.62206},
    {   9.93500,  6.43849,  3.62149},
    {   1.68791,  6.46940,  3.62222},
    {   5.79045, -0.66896,  3.62107},
    {   8.04677,  0.17319,  2.25276},
    {   9.91728,  1.68940,  3.62190},
    {   9.34215, -2.08642,  1.36053},
    {   8.03146, -4.33707,  2.25136},
    {   3.52565, -4.31881,  2.25154},
    {   5.78092, -3.47745,  3.62053},
    {   2.22892, -2.05789,  1.35978},
  }
};

